classdef CKeuningHydroModel
    % Class encapsulates hydodynamic force calculation for given Maindimensions
    % See Jost Kempers BE Thesis
    
    %% general class properties
    properties
        MainDimensions
        Name
        FileType
        
        % Environment Values
        rho
        nu_kin
        gravity
    end
    
    
    methods
        %% Constructor, loads main dimension from JSON file
        function obj = CKeuningHydroModel(JSONstruct, DEFAULTstruct, DEFAULTconstants)
            
            % read file name and file type
            obj.Name = CHelperFct.getFromStruct(JSONstruct, 'Name', DEFAULTstruct.Name);
            obj.FileType = CHelperFct.getFromStruct(JSONstruct, 'FileType', DEFAULTstruct.FileType);
            
            obj.MainDimensions = JSONstruct.Dimensions;
            
            % read default values for physical and other constants and write to class members
            obj.rho = DEFAULTconstants.densityWater;
            obj.nu_kin = DEFAULTconstants.viscosityWater;
            obj.gravity = DEFAULTconstants.gravity;
                       
        end
        
        %% get total forces as [FX,FY,FZ, Mx',My',Mz']
        function forces = getForces(obj, speed,leeway, heel, rudder)
            
            Fn  = speed/(obj.gravity*obj.MainDimensions.Lwl)^0.5;
            
            %Calculate FL in DLS-coordinate sys
            FH = obj.getFHs(speed, heel, leeway, rudder);
            FL = FH.FHk + FH.FHr;
            FS = 0;       % model tests free to sink and trim
            FD = obj.getRfh_phi(speed,heel) + obj.getRrh(Fn) + obj.getRvk(speed)+...
                obj.getRvr(speed) + obj.getRrk(Fn) + obj.getDRrh_phi(Fn,heel)+...
                obj.getDRrk_phi(Fn,heel) + obj.getDRr_phibeta(speed,heel,FL);
            
            %             % DEBUG
            %             Rfh_phi = obj.getRfh_phi(speed,heel)            %frictional resistance of Hull, including heel correction
            %             Rrh = obj.getRrh(Fn)                            %calculates residuary resistance of hull in [N]
            %             Rvk = obj.getRvk(speed)                         % viscous resistance of keel in [N]
            %             Rvr = obj.getRvr(speed)                         % viscous resistance of rudder in [N]
            %             Rrk = obj.getRrk(Fn)                            % keel residuary resistance in [N]
            %             DRrh_phi = obj.getDRrh_phi(Fn,heel)             % delta resiuary resistance of hull due to heel in [N]
            %             DRrk_phi = obj.getDRrk_phi(Fn,heel)             % delta residuary resistance keel due to heel in [N]
            %             DRr_phibeta = obj.getDRr_phibeta(speed,heel,FL) % induced resistance in [N]
                        
            % Calculate MD, not quite clear if the righting moment acts around the x'-axis
            MD = obj.getRM(heel) + obj.getDMx_H( leeway, FH) + obj.getDMx_Crew(heel);
            
            %             DEBUG
            %             RM = obj.getRM(heel)
            %             DMx_H = obj.getDMx_H( leeway, FH)
            %             DMx_Crew = obj.getDMx_Crew(heel)
                       
            % My', assumed to be zero, because at tank testing the model was free to sink and trim
            ML = 0;
            
            %Calculate MS if ap
            MS  =  obj.getMZ_H(FH);              % obj.getMZM_phi(u, phi, beta) +    %  Munk moment disabled
            
            % transfer forces to XYZ coordinate system
            forces(1:3) = CHelperFct.rotx(heel) * [FD FL FS].';
            forces(4:6) = CHelperFct.rotx(heel) * [MD ML MS].';
            forces = forces.';
            
        end %getForces()
        
    end % methods
    
end

