
classdef CYruAeroModel
    %Class encapsulates aerodynamic force calculation for given
    %maindimensions and sail parameters
    %   Detailed explanation goes here
    
    properties
        Name
        FileType
        BAD_P
        HBI_P
        E_P
        I__BAD_P_
        I_J
        XG_E
        se
        cD0flatExponent
        dcD0dTwist
        dvcedTwist
        AWA
        cL
        cD0
        cS
        mD
        mL
        mS
        
        rhoA = 1.25;
        
        cLinterp
        cD0interp
        cSinterp
        mDinterp
        mLinterp
        mSinterp

        areaRef
        span
        
    end
    
    methods
        %% Constructor, reads properties from JSON struct
        function obj = CYruAeroModel(I,J,P,E)
            
            % interactively get the file name and path
            [file, path] = uigetfile('*.json', 'Select an aero data json file', MultiSelect= 'off');
            if path == 0; return; end   % file dialog aborted

            % read file content into a JSON parser and generate a struct
            JSONstruct = jsondecode(CHelperFct.removeJSONComments( fileread(fullfile(path, file))) );
                      
            % geometric ratios - THESE ARE NOT USED FOR FLOW CALCULATIONS
            obj.BAD_P = JSONstruct.BAD_P;
            obj.HBI_P = JSONstruct.HBI_P;
            obj.E_P = JSONstruct.E_P;
            obj.I__BAD_P_ = JSONstruct.I__BAD_P_;
            obj.I_J = JSONstruct.I_J;
            obj.E_P = JSONstruct.E_P;

            obj.areaRef = (I*J+P*E)/2;
            obj.span = (1 + obj.BAD_P + obj.HBI_P) * P;
            
            % aerodynamic coefficients
            obj.se = JSONstruct.SE;
            obj.XG_E = JSONstruct.XG_E;
            obj.cD0flatExponent = JSONstruct.cD0flatExponent;
            obj.dcD0dTwist = JSONstruct.dcD0dTwist;
            obj.dvcedTwist = JSONstruct.dvcedTwist;
            obj.AWA = JSONstruct.AWA;
            obj.cL = JSONstruct.cL;
            obj.cD0 = JSONstruct.cD0;
            obj.cS = JSONstruct.cS;
            obj.mD = JSONstruct.mD;
            obj.mL = JSONstruct.mL;
            obj.mS = JSONstruct.mS;
                       
            % interpolators
            obj.cLinterp = griddedInterpolant(obj.AWA, obj.cL,'makima');
            obj.cD0interp = griddedInterpolant(obj.AWA, obj.cD0,'makima');
            obj.cSinterp = griddedInterpolant(obj.AWA, obj.cS,'makima');
            obj.mDinterp = griddedInterpolant(obj.AWA, obj.mD,'makima');
            obj.mLinterp = griddedInterpolant(obj.AWA, obj.mL,'makima');
            obj.mSinterp = griddedInterpolant(obj.AWA, obj.mS,'makima');
                     
        end % constructor
        

        %% function interpolates sail coefficients for given AWA
        function [cL, cD0, cS, mD, mL, mS] = getCoefficients( obj , thisAWA )
            cL = obj.cLinterp(thisAWA);
            cD0 = obj.cD0interp(thisAWA);
            cS = obj.cSinterp(thisAWA);
            mD = obj.mDinterp(thisAWA);
            mL = obj.mLinterp(thisAWA);
            mS = obj.mSinterp(thisAWA);
        end % getCoefficients()
        
        %% function calculates aerodynamic forces as [FX,FY,FZ, Mx',My',Mz']
        function forces = getForces( obj, speed, heel, leeway, flat, twist, TWS, TWA)
                    
            %apparend wind(measured with respect to the boats centerplane)
            thisAWS = ((TWS*cosd(TWA)+speed*cosd(leeway))^2+((TWS*sind(TWA)-speed*sind(leeway))*cosd(heel))^2)^0.5;
            thisAWA = atan2d((TWS*sind(TWA)-speed*sind(leeway))*cosd(heel), TWS*cosd(TWA)+speed*cosd(leeway));
            
            % sail coefficients
            [thisCL, thisCD0, thisCS, thisMD, thisML, thisMS] = obj.getCoefficients(thisAWA ); 
            
            % correction for CD0-dependency on flat
            thisCD0 = thisCD0 * flat^obj.cD0flatExponent;

            % flattend sail coefficients, drag coefficient includes induced drag, induced drag includes correction for twist
            CL = thisCL * flat;
            CD = thisCD0 * flat^obj.cD0flatExponent + CL^2*obj.areaRef/(pi * (obj.se*obj.span)^2) + obj.dcD0dTwist * twist;
            CS = thisCS * flat;
            MD = thisMD * flat;
            ML = thisML * flat;
            MS = thisMS * flat;
            
            % forces in DLS_aero coordinate system
            FD = CD * 0.5 * obj.rhoA * thisAWS^2 * obj.areaRef;
            FL = CL * 0.5 * obj.rhoA * thisAWS^2 * obj.areaRef;
            FS = CS * 0.5 * obj.rhoA * thisAWS^2 * obj.areaRef;
            QD = MD * 0.5 * obj.rhoA * thisAWS^2 * obj.areaRef^1.5;
            QL = ML * 0.5 * obj.rhoA * thisAWS^2 * obj.areaRef^1.5;
            QS = MS * 0.5 * obj.rhoA * thisAWS^2 * obj.areaRef^1.5;
            
            % forces in x', y', z' coordinate system (boat fixed coord sys)
            forces(1:3) = CHelperFct.rotz(-thisAWA) * CHelperFct.rotx(180) * [FD FL FS].';
            forces(4:6) = CHelperFct.rotz(-thisAWA) * CHelperFct.rotx(180) * [QD QL QS].';

            % apply TWIST for the moment around x'- and y'-axis (not around z'-axis !!)
            forces(4) = forces(4) - forces(2) * obj.dvcedTwist * twist * obj.span;
            forces(5) = forces(5) - forces(1) * obj.dvcedTwist * twist * obj.span;
           
            % transform forces to XYZ-coordinate system 
            % coord sys: ship longitudinal aft (x), normal to water plane (z)
            %
%             forces(1:3) = CHelperFct.rotz(-leeway) * CHelperFct.rotx(heel) * forces(1:3).';
%             forces(4:6) = CHelperFct.rotz(-leeway) * CHelperFct.rotx(heel) * forces(4:6).';
            forces(1:3) = CHelperFct.rotx(heel) * forces(1:3).';
            forces(4:6) = CHelperFct.rotx(heel) * forces(4:6).';            
            forces = forces.';
                       
        end % getForces()
        
    end % methods
    
end % classdef

