classdef CAeroModell
    %% Class encapsulates aerodynamic force calculation
    
    properties
        %%
        xTack = 0.0;           % longitudinal distance from WT dynamometer origin to mainsail tack point             FULL SCALE
        zTack = 0.9;           % vertical     distance from WT dynamometer origin to mainsail tack point (HBI+BAD)   FULL SCALE
        kpp = 0.015;        % parasitic profile drag coeff
        lref = 4.2;           % reference length for normalization
        span = 6.70;        % distance water plane headboard (for normalization
        rhoA = 1.25;        % air density in kg/m³
        aref = 11.63;        % =P*E/2+I*J/2;   P=6.265, E=2.50 geschätzt, J = 2.0, I = 3.815
        
        % wt coeffs table
        %         awa = [0;3;10; 12; 14; 16; 18; 20; 22; 24; 26; 28; 30; 32; 34; 36; 38; 40; 42; 44; 46; 48; 50];
        %         cl = [0;0;0.57126254836769086; 0.72224746725804911; 0.8714329243856519; 1.0097950397916455; 1.1293055391985789; 1.2299968173860818; 1.3171957834562318; 1.3914504771049567; 1.4499326518166629; 1.4907754968740812; 1.516060971000333; 1.5304621488487469; 1.5342702903205294; 1.52676933730577; 1.5117822876510345; 1.4976783053327212; 1.4903767644002488; 1.4851859294291845; 1.4749831145184895; 1.4578722896165166; 1.4371922677808167];
        %         cd0 = [0.08;0.08;0.079468700443311016; 0.08244879541074876; 0.085878387108615256; 0.0908227957897855; 0.098406411211235; 0.10946076197333686; 0.12470273522530416; 0.14455741181662241; 0.16836128383833393; 0.19460419119933639; 0.2219858482218568; 0.24990296525280165; 0.27656486246671613; 0.29998522211409462; 0.32121384811240655; 0.34434142195814432; 0.37205461756230657; 0.40088266559744795; 0.42596992648180704; 0.44608804142345027; 0.4636416143497385];
        %         vce = [0.27;0.27;0.27318495213763028; 0.28386850658579704; 0.29366578271057114; 0.30152013559603547; 0.30683712858595685; 0.30974302196543485; 0.31066159369865487; 0.30986371660680412; 0.30727675895070916; 0.30290733602107128; 0.29724676968121655; 0.2911924220457609; 0.28536619148401587; 0.28028689840192439; 0.27600493257667963; 0.27210151938679489; 0.26821942685703426; 0.26433000839710413; 0.26048519049368229; 0.256697198759073; 0.25293849574202631];
        %         lce = [-0.029;-0.029; -0.039639226482090735; -0.047295721192645621; -0.054248156699649594; -0.059455560182228882; -0.062066411787590543; -0.062116508795715453; -0.060650452916587425; -0.058695546671023738; -0.056816681003851205; -0.0550527437649757; -0.053139980410581072; -0.050841761989750413; -0.048045471590271768; -0.044674449837098895; -0.040771530037392326; -0.036499225359005849; -0.0320166258658926; -0.027448194355741064; -0.0229065875301298; -0.018432183906393451; -0.013992970597958268];
        
        % old ORC
        awa = [11.2500720000000;15;18.7499280000000;22.4998560000000;26.2497840000000;30;33.7499280000000;37.4998560000000;41.2497840000000;45;48.7499280000000;52.4998560000000;56.2497840000000;60;63.7499280000000;67.4998560000000;71.2500720000000;75;78.7499280000000;82.4998560000000;86.2500720000000;90];
        cl = [0.514412320672134;0.775072171903467;1.04074644983115;1.28450105387050;1.48544132979824;1.63243052293866;1.72446958076362;1.76802999652650;1.77317768784605;1.75045387612537;1.70908918797283;1.65633507864050;1.59749318307984;1.53622927877200;1.47497838106485;1.41526357895984;1.35798586846313;1.30364039383879;1.25243425345440;1.20440379259216;1.15947398720152;1.11751672197051];
        cd0 = [0.0744612364627614;0.0902885106225079;0.110548075560943;0.134685496819578;0.162515925518142;0.194012421914793;0.229207417342534;0.268163293317602;0.310937087241655;0.357568583729556;0.408051660649877;0.462343951646539;0.520342643484908;0.581884243672366;0.646718847359557;0.714534416488506;0.784942481109448;0.857463176451377;0.931563757992945;1.00664697548749;1.08207563619152;1.15716596040198];
        vce = [0.378479745560774;0.375905623362219;0.372925232330873;0.370081352920870;0.367588123393183;0.365538490269433;0.363974989883409;0.362916903173993;0.362373576104821;0.362350478892254;0.362852456020349;0.363885573983123;0.365458310744724;0.367582499232761;0.370273243196765;0.373550444624230;0.377439078102830;0.381968712754486;0.387176023737733;0.393104513003091;0.399806382952460;0.407341712314021];
        lce = [0.0131144622497101;0.0185723204580403;0.0188449259215508;0.0159035418982990;0.0108263988975679;0.00425083398198546;-0.00342108860215642;-0.0119252534174949;-0.0210801633953448;-0.0307579930782082;-0.0408636575332399;-0.0513281800122609;-0.0620986207190820;-0.0731349379474523;-0.0844033672773817;-0.0958792753127181;-0.107543100323748;-0.119376442347216;-0.131366592524056;-0.143502330002990;-0.155775278874510;-0.168175760014780];
        
        clInterp;
        cd0Interp;
        vceInterp;
        lceInterp;
        
        
    end
    
    methods
        %% Constructor
        function obj = CAeroModell()
            obj.clInterp = griddedInterpolant(obj.awa, obj.cl,'linear');
            obj.cd0Interp = griddedInterpolant(obj.awa, obj.cd0,'pchip');
            obj.vceInterp = griddedInterpolant(obj.awa, obj.vce,'makima');
            obj.lceInterp = griddedInterpolant(obj.awa, obj.lce,'makima');
        end
        
        %% coefficient interpolation from tables
        function [awa, thisCl, thisCd0, thisLce,thisVce] = getCoefficients( obj, awa )
            %function interpolates sail coefficients for given AWA
            thisCl = obj.clInterp(awa);
            thisCd0 = obj.cd0Interp(awa);
            thisLce = obj.lceInterp(awa);
            thisVce = obj.vceInterp(awa);
        end
        
        %% function calculates aerodynamic forces
        function Freturn = getForces( obj, velo, heel, leeway, flat, TWS, TWA)
            %function calculates aerodynamic forces
            
            %apparend wind(measured with respect to the boats centerplane)
            AWS = ((TWS*cosd(TWA)+velo*cosd(leeway))^2+((TWS*sind(TWA)-velo*sind(leeway))*cosd(heel))^2)^0.5;
            AWA = atan2d((TWS*sind(TWA)-velo*sind(leeway))*cosd(heel), TWS*cosd(TWA)+velo*cosd(leeway));
            AWA = AWA + (AWA < 0) * 360;
            
            %sail coefficients
            [AWA,thisCl,thisCd0, thisLce,thisVce] = obj.getCoefficients(AWA);
            
            % derived current aero coefficients
            CE = obj.kpp + ( 1/(pi * ( (obj.span*(1 + 0.1 * (1 - min(max((abs(AWA)-30)/60,0),1))))^2 / obj.aref )));
            CL = thisCl * flat;
            CD = 0.006 + thisCd0 * flat  + CE * CL.^2;    % TODO my idea
            VCE = thisVce * obj.span;
            LCE = thisLce * obj.lref - obj.xTack;
            
            %Calculate FX
            FX = 0.5 * obj.rhoA * AWS^2 * obj.aref * (CL * sind(AWA) - CD * cosd(AWA));
            %Calculate FH, perpendicluar to span
            FH = 0.5 * obj.rhoA * AWS^2 * obj.aref * (CL * cosd(AWA) + CD * sind(AWA));
            % Calc FY
            FY = FH * cosd(heel);
            % FZ vertical direction
            FZ = -FH * sind(heel);
            %Calculate Mx
            MX = FH * VCE;
            %Calculate MY
            MY = -FX * VCE * cosd(heel) + FZ * LCE;
            %Calculate MZ
            MZ = FY * LCE - FX * VCE * sind(heel);
            
            Freturn = [FX; FY; FZ; MX; MY; MZ];
            
        end  % fct getForces
        
    end % methods
    
end % classdefinition CAeroModell

